#!/usr/local/bin/bash

# This script demonstrates a potential pf bug on pfSense 24.11+ on x86 hardware.
#
# The bug is that if a tag is applied to a packet in multiple nested anchors,
# it will not be applied in a tagged pass rule

# --- Configuration ---
DESTINATION_TEST_IP="208.123.73.78"
TABLE_NAME="example_dst_table"
TAG_NAME="example_tag"
ANCHOR_PARENT="userrules"
ANCHOR1="bug_anchor1"
ANCHOR2="bug_anchor2"
RULES_FILE="/tmp/bug_rules.txt"

# --- Cleanup previous runs ---
# Clean up to ensure a fresh state.
# These commands will fail if the anchors/tables don't exist, which is fine.
pfctl -a $ANCHOR_PARENT/$ANCHOR1 -F all 2>/dev/null
pfctl -a $ANCHOR_PARENT/$ANCHOR2 -F all 2>/dev/null
pfctl -t $TABLE_NAME -T flush 2>/dev/null

echo "Cleaned up any previous rules and tables."

# 1. Create an IP address table
echo "1. Creating table '$TABLE_NAME' with ${DESTINATION_TEST_IP}..."
pfctl -t $TABLE_NAME -T add $DESTINATION_TEST_IP
echo "Table '$TABLE_NAME' created."
pfctl -t $TABLE_NAME -T show

# 2. Create the anchor rules in a temporary file
echo "2. Creating temporary rules file at $RULES_FILE..."
if [[ "$1" == "single-anchor" ]]; then
cat > $RULES_FILE <<EORULES
# Demonstrate that a single anchor works
anchor "$ANCHOR1" {
    match in log to <$TABLE_NAME> tag "$TAG_NAME" label "$ANCHOR1"
}
EORULES
else
cat > $RULES_FILE <<EORULES
# In each anchor, create a match and log rule that applies the tag.
anchor "$ANCHOR1" {
    match in log to <$TABLE_NAME> tag "$TAG_NAME" label "$ANCHOR1"
}
anchor "$ANCHOR2" {
    match in log to <$TABLE_NAME> tag "$TAG_NAME" label "$ANCHOR2"
}
EORULES
fi

# Load the rules into the parent anchor
echo "3. Loading rules into parent anchor '${ANCHOR_PARENT}/*'"
pfctl -a $ANCHOR_PARENT -f $RULES_FILE

echo "--- Rules Loaded ---"
echo "The following rules have been loaded under the '$ANCHOR_PARENT' anchor:"
pfctl -a $ANCHOR_PARENT -s rules

echo "--- Manual Step Required ---"
echo "Create an allow rule on your LAN interface in pfSense, and allow any traffic tagged with \"$TAG_NAME\"."
echo "Enable logging on this rule to help with demonstrating the behaviour below."
echo "----------------------------"

echo "--- Setup Complete ---"
echo "To test this, you must generate traffic from your LAN to $DESTINATION_TEST_IP."
echo "For example, from a client computer on your LAN, run:"
echo "ping $DESTINATION_TEST_IP"

echo "To observe the bug, check the firewall logs on pfSense."
echo "You can use the following command to filter for the relevant log entries:"
echo "tail -F /var/log/filter.log | grep $DESTINATION_TEST_IP"

echo "--- Expected Behavior (Demonstrating Bug) ---"
echo "When you generate traffic, you should see log entries from the firewall rule labels '$ANCHOR1' and '$ANCHOR2'."
echo "This confirms that the traffic is being matched and tagged in both anchors."
echo "However, you will NOT see any log entries for the pass rule you created in the manual step above."
echo "This demonstrates the bug: the 'pass' rule is not being hit, and the traffic will likely be blocked by a default deny rule instead of being passed."
